<?php
// time_comparison.php - Compara la hora de la base de datos con el endpoint externo

// Reportar errores para depuración
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Encabezados para formato JSON
header('Content-Type: application/json; charset=utf-8');

// Conexión a la base de datos - Ajusta estos valores según tu configuración
require_once __DIR__ . '/../con/config.php'; // Asumiendo que este archivo tiene la conexión a la DB

// Función para obtener la hora desde el endpoint externo
function getExternalTime() {
    $url = 'https://panel.digital-tv.net/hora.php';
    $ch = curl_init($url);
    
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Considera activar en producción
    
    $response = curl_exec($ch);
    
    if(curl_errno($ch)) {
        return ['error' => 'Error en CURL: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    $data = json_decode($response, true);
    return $data;
}

try {
    // Configurar zona horaria para PHP
    date_default_timezone_set('America/Argentina/Mendoza');
    
    // Obtener hora del servidor PHP
    $php_time = [
        'datetime' => date('Y-m-d H:i:s'),
        'timestamp' => time(),
        'timezone' => date_default_timezone_get()
    ];
    
    // Obtener hora del endpoint externo
    $external_time_response = getExternalTime();
    $external_time = [];
    
    if (isset($external_time_response['datetime'])) {
        $external_time['original'] = $external_time_response['datetime'];
        $external_time['parsed'] = date('Y-m-d H:i:s', strtotime($external_time_response['datetime']));
        $external_time['timestamp'] = strtotime($external_time_response['datetime']);
    } else {
        $external_time['error'] = 'No se pudo obtener la hora del endpoint externo';
        if (isset($external_time_response['error'])) {
            $external_time['error_details'] = $external_time_response['error'];
        }
    }
    
    // Obtener hora de la base de datos
    $db_time = [];
    
    if (isset($conn) && $conn) {
        $result = $conn->query("SELECT NOW() as db_time, UNIX_TIMESTAMP() as db_timestamp");
        if ($result) {
            $row = $result->fetch_assoc();
            $db_time['datetime'] = $row['db_time'];
            $db_time['timestamp'] = $row['db_timestamp'];
            
            // Obtener zona horaria de MySQL
            $timezone_result = $conn->query("SELECT @@system_time_zone as timezone");
            if ($timezone_result) {
                $timezone_row = $timezone_result->fetch_assoc();
                $db_time['timezone'] = $timezone_row['timezone'];
            }
        } else {
            $db_time['error'] = 'Error al consultar la base de datos: ' . $conn->error;
        }
    } else {
        $db_time['error'] = 'No hay conexión a la base de datos disponible';
    }
    
    // Calcular diferencias entre las horas
    $differences = [];
    
    if (!empty($php_time['timestamp']) && !empty($db_time['timestamp'])) {
        $differences['php_vs_db'] = [
            'seconds' => $php_time['timestamp'] - $db_time['timestamp'],
            'formatted' => gmdate('H:i:s', abs($php_time['timestamp'] - $db_time['timestamp']))
        ];
    }
    
    if (!empty($php_time['timestamp']) && !empty($external_time['timestamp'])) {
        $differences['php_vs_external'] = [
            'seconds' => $php_time['timestamp'] - $external_time['timestamp'],
            'formatted' => gmdate('H:i:s', abs($php_time['timestamp'] - $external_time['timestamp']))
        ];
    }
    
    if (!empty($db_time['timestamp']) && !empty($external_time['timestamp'])) {
        $differences['db_vs_external'] = [
            'seconds' => $db_time['timestamp'] - $external_time['timestamp'],
            'formatted' => gmdate('H:i:s', abs($db_time['timestamp'] - $external_time['timestamp']))
        ];
    }
    
    // Simular verificación de vencimiento
    $test_cases = [];
    
    if (!empty($db_time['datetime']) && !empty($external_time['parsed'])) {
        // Caso 1: Usando la hora de la DB como "hoy" y la hora del endpoint como "vencimiento"
        $db_date = new DateTime($db_time['datetime']);
        $external_date = new DateTime($external_time['parsed']);
        
        $test_cases['db_hoy_external_vencimiento'] = [
            'resultado' => ($external_date >= $db_date) ? 'Cuenta activa' : 'Cuenta expirada',
            'diferencia_dias' => $db_date->diff($external_date)->format('%R%a días')
        ];
        
        // Caso 2: Usando la hora del endpoint como "hoy" y la hora de la DB como "vencimiento"
        $test_cases['external_hoy_db_vencimiento'] = [
            'resultado' => ($db_date >= $external_date) ? 'Cuenta activa' : 'Cuenta expirada',
            'diferencia_dias' => $external_date->diff($db_date)->format('%R%a días')
        ];
    }
    
    // Preparar respuesta
    $response = [
        'php_time' => $php_time,
        'db_time' => $db_time,
        'external_time' => $external_time,
        'differences' => $differences,
        'test_cases' => $test_cases
    ];
    
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    
} catch (Exception $e) {
    echo json_encode(['error' => 'Error en el script: ' . $e->getMessage()]);
} finally {
    // Cerrar la conexión a la base de datos si está disponible
    if (isset($conn) && $conn) {
        $conn->close();
    }
}
?>