<?php
// Incluir archivo de configuración de la base de datos
require_once __DIR__ . '/../con/config.php';

// Desactivar la salida de errores para producción
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(0); // Solo reportar errores al log, no al output

header('Content-Type: application/json');

// Configuraciones
$secret_key = "3e1g9x52z15b2u54";
$private_key_path = '/public_html/panel_api/private_key.pem';
$cipher = "AES-128-ECB";
$nonce_tolerance = 300; // 5 minutos de tolerancia

// Verificar método POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(["error" => "Método no permitido"]);
    exit;
}

// Verificar credenciales
$email = isset($_POST['email']) ? filter_var($_POST['email'], FILTER_SANITIZE_EMAIL) : null;
$password = $_POST['password'] ?? null;
$encrypted_nonce = $_POST['nonce'] ?? null;

// Verificar que se proporcionaron todos los campos requeridos
if (!$email || !$password || !$encrypted_nonce) {
    http_response_code(400);
    echo json_encode(["error" => "Se requieren email, password y nonce"]);
    exit;
}

// Verificar credenciales en la base de datos
$stmt = $conn->prepare("SELECT email, password, vencimiento FROM users WHERE email = ? LIMIT 1");
$stmt->bind_param("s", $email);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

if (!$user || $user['password'] !== $password) {
    http_response_code(401);
    echo json_encode(["error" => "Credenciales inválidas"]);
    exit;
}

// Verificar si la cuenta no ha expirado
$hoy = new DateTime();
$vencimiento = new DateTime($user['vencimiento']);

if ($vencimiento < $hoy) {
    http_response_code(401);
    echo json_encode(["error" => "Cuenta expirada"]);
    exit;
}

// Cerrar la conexión a la base de datos
$conn->close();

// Procesar el nonce
$private_key_content = file_get_contents($private_key_path);
if ($private_key_content === false) {
    http_response_code(500);
    echo json_encode(["error" => "Error del servidor - archivo de clave no encontrado"]);
    exit;
}

$private_key = openssl_pkey_get_private($private_key_content);
if (!$private_key) {
    http_response_code(500);
    echo json_encode(["error" => "Error del servidor - clave inválida"]);
    exit;
}

// Decodificar y desencriptar el nonce
$decoded_nonce = base64_decode($encrypted_nonce);
if ($decoded_nonce === false) {
    // No necesitamos liberar la clave manualmente en PHP 8+, se maneja automáticamente
    http_response_code(400);
    echo json_encode(["error" => "Formato de nonce inválido"]);
    exit;
}

$nonce = null;
if (!openssl_private_decrypt($decoded_nonce, $nonce, $private_key)) {
    http_response_code(400);
    echo json_encode(["error" => "Nonce inválido"]);
    exit;
}

// Verificar que el nonce sea válido (un timestamp reciente)
$nonce_time = (int)$nonce;
$current_time = time();
if (abs($current_time - $nonce_time) > $nonce_tolerance) {
    http_response_code(400);
    echo json_encode(["error" => "Nonce expirado"]);
    exit;
}

// No necesitamos openssl_free_key(), PHP lo maneja automáticamente desde PHP 8.0

// Generar una clave temporal basada en el nonce
$temp_key = hash('sha256', $nonce, true);

// Encriptar la clave AES con la clave temporal
$encrypted_key = openssl_encrypt(
    $secret_key, 
    $cipher, 
    substr($temp_key, 0, 16), 
    OPENSSL_RAW_DATA
);

if ($encrypted_key === false) {
    http_response_code(500);
    echo json_encode(["error" => "Error de encriptación"]);
    exit;
}

// Devolver la clave AES encriptada
echo json_encode(["encrypted_key" => base64_encode($encrypted_key)]);
?>